# Sender - Complete Implementation Guide

## Quick Start

### 1. Installation Steps

1. **Upload files** to your web server
2. **Create database** in MySQL
3. **Navigate to** `http://yourdomain.com/install/`
4. **Follow the installation wizard**:
   - Check system requirements
   - Configure database connection
   - Enter WhatsApp API credentials
   - Create admin account
5. **Set up cron job** for queue processing

### 2. Cron Job Setup

Add this to your crontab:
```bash
*/1 * * * * php /path/to/sender-app/cron/process-queue.php >> /path/to/sender-app/storage/logs/cron.log 2>&1
```

For daily cleanup:
```bash
0 2 * * * php /path/to/sender-app/cron/cleanup.php >> /path/to/sender-app/storage/logs/cleanup.log 2>&1
```

### 3. WhatsApp Webhook Configuration

**Webhook URL:** `https://yourdomain.com/public/webhook/whatsapp`
**Verify Token:** Use the token generated during installation

Configure this in your Meta for Developers dashboard.

## File Structure Created

```
sender-app/
├── config/               # Configuration files
│   ├── app.php          # App settings
│   ├── database.php     # DB connection
│   ├── whatsapp.php     # WhatsApp API
│   └── payment.php      # Payment & pricing (CFA)
├── core/                # MVP Core framework
│   ├── Database.php     # Database class
│   ├── Router.php       # Routing system
│   ├── Controller.php   # Base controller
│   ├── Model.php        # Base model
│   ├── View.php         # View rendering
│   └── Session.php      # Session management
└── install/             # Installation system
    ├── index.php        # Installation wizard
    └── install.sql      # Database schema
```

## Next Steps for Development

### Phase 1: Core Models (PRIORITY)
Create these models in `app/models/`:

1. **User.php** - User authentication & management
2. **Contact.php** - Contact database
3. **Template.php** - Message templates
4. **Campaign.php** - Campaign management
5. **Message.php** - Message tracking
6. **Payment.php** - Payment processing
7. **Subscription.php** - Subscription management

### Phase 2: Controllers
Create these controllers in `app/controllers/`:

1. **AuthController.php** - Login, register, password reset
2. **DashboardController.php** - Main dashboard
3. **ContactController.php** - Contact CRUD, import, export
4. **TemplateController.php** - Template management
5. **CampaignController.php** - Campaign builder
6. **InboxController.php** - Conversations & replies
7. **AnalyticsController.php** - Reports & metrics
8. **SettingsController.php** - Account & system settings
9. **BillingController.php** - Subscriptions & payments

### Phase 3: Views
Create views in `app/views/` for each controller.

### Phase 4: WhatsApp Integration
Create `app/services/WhatsAppService.php`:
- Send messages
- Upload media
- Handle webhooks
- Manage templates

### Phase 5: Queue Processing
Implement `cron/process-queue.php`:
- Fetch pending messages
- Send via WhatsApp API
- Update status
- Handle retries

### Phase 6: Payment Integration
Create payment gateways in `app/services/`:
- MTNMomoService.php
- OrangeMoneyService.php
- BankTransferService.php

## Key Features to Implement

### Contact Management
- [x] Database schema created
- [ ] Import from CSV/Excel
- [ ] Extract from WhatsApp groups (using WhatsApp Business API)
- [ ] Contact segmentation
- [ ] Export contacts

### Template Management
- [x] Database schema created
- [ ] Create templates
- [ ] Variable substitution
- [ ] Media support (header images/videos)
- [ ] Template approval workflow

### Campaign System
- [x] Database schema created
- [ ] Campaign builder UI
- [ ] Recipient selection (groups, segments, all)
- [ ] Schedule campaigns
- [ ] Queue messages
- [ ] Monitor progress

### Messaging
- [x] Database schema created
- [ ] Queue system
- [ ] Bulk sending
- [ ] Status tracking (sent, delivered, read)
- [ ] Error handling & retries
- [ ] Rate limiting

### Inbox/Conversations
- [x] Database schema created
- [ ] Incoming message handling
- [ ] Conversation threads
- [ ] Reply functionality
- [ ] Assignment to users

### Analytics
- [x] Database schema created
- [ ] Dashboard metrics
- [ ] Campaign reports
- [ ] Delivery statistics
- [ ] Export reports

### Billing
- [x] Database schema created
- [x] Pricing configured (CFA)
- [ ] Subscription management
- [ ] Payment gateway integration
- [ ] Invoice generation
- [ ] Usage tracking

## WhatsApp API Integration

### Required API Calls

1. **Send Template Message**
```php
POST https://graph.facebook.com/v21.0/{phone-number-id}/messages
Headers: Authorization: Bearer {access-token}
Body: {
  "messaging_product": "whatsapp",
  "to": "{recipient-phone}",
  "type": "template",
  "template": {
    "name": "{template-name}",
    "language": { "code": "en" }
  }
}
```

2. **Upload Media**
```php
POST https://graph.facebook.com/v21.0/{phone-number-id}/media
Headers: Authorization: Bearer {access-token}
Form-data: file={media-file}
```

3. **Send Media Message**
```php
POST https://graph.facebook.com/v21.0/{phone-number-id}/messages
Body: {
  "messaging_product": "whatsapp",
  "to": "{recipient-phone}",
  "type": "image|video|document",
  "image|video|document": {
    "id": "{media-id}"
  }
}
```

### Webhook Handling

Listen for:
- Message status updates (sent, delivered, read)
- Incoming messages
- Template status changes

## Security Checklist

- [x] Password hashing (bcrypt)
- [x] PDO prepared statements
- [ ] CSRF protection implementation
- [ ] XSS prevention in views
- [ ] Rate limiting on API endpoints
- [ ] File upload validation
- [ ] SSL certificate (required for webhooks)
- [ ] Environment variables for secrets

## Testing Checklist

### Before Going Live
- [ ] Test database connections
- [ ] Test WhatsApp API connection
- [ ] Send test messages
- [ ] Test webhook receiving
- [ ] Test queue processing
- [ ] Test payment flows
- [ ] Test all user roles
- [ ] Test rate limiting
- [ ] Load testing with bulk sends

## Deployment Checklist

- [ ] Set environment to production
- [ ] Disable debug mode
- [ ] Configure error logging
- [ ] Set up SSL certificate
- [ ] Configure cron jobs
- [ ] Set file permissions (storage: 755, uploads: 755)
- [ ] Configure database backups
- [ ] Set up monitoring
- [ ] Document admin credentials
- [ ] Create backup & recovery plan

## Support & Documentation

### Essential Links
- WhatsApp Cloud API Docs: https://developers.facebook.com/docs/whatsapp/cloud-api
- Meta for Developers: https://developers.facebook.com/
- WhatsApp Business Manager: https://business.facebook.com/wa/manage/
- API Testing Tool: https://developers.facebook.com/tools/explorer/

### Common Issues & Solutions

**Issue**: Webhook not receiving messages
**Solution**: Ensure URL is HTTPS, verify token matches, check firewall rules

**Issue**: Messages not sending
**Solution**: Check WhatsApp API credentials, verify phone number is registered, check rate limits

**Issue**: Queue not processing
**Solution**: Verify cron job is running, check logs in storage/logs/

**Issue**: Payment not reflecting
**Solution**: Check payment gateway configuration, verify webhooks for MTN/Orange

## Customization

### Adding New Features
1. Create model in `app/models/`
2. Create controller in `app/controllers/`
3. Add routes in `public/index.php`
4. Create views in `app/views/`

### Modifying Pricing
Edit `config/payment.php` to adjust:
- Plan prices (in CFA)
- Plan features
- Message credit costs
- Trial period

### Changing Themes
Edit CSS in `public/assets/css/style.css`

## Performance Optimization

- Enable OPcache for PHP
- Use connection pooling for database
- Implement Redis for caching (optional)
- Optimize queue batch size
- Use CDN for static assets
- Compress images before upload
- Database indexing (already in schema)

## Backup Strategy

### Database
```bash
mysqldump -u username -p sender_db > backup_$(date +%Y%m%d).sql
```

### Files
```bash
tar -czf backup_files_$(date +%Y%m%d).tar.gz /path/to/sender-app/public/uploads/
```

## Monitoring

Monitor these metrics:
- Queue processing time
- Message delivery rate
- API error rate
- Database query performance
- Disk space (uploads)
- Memory usage
- Cron job execution

## License & Credits

This is a custom-built WhatsApp bulk messaging platform inspired by respond.io.
Built with PHP using MVP architecture pattern.

## Getting Help

For issues or questions:
1. Check the documentation
2. Review error logs in `storage/logs/`
3. Verify WhatsApp API status
4. Check Meta for Developers community

---

**Remember**: Always test in a development environment before deploying to production!
