# 🚀 Sender - Quick Start Guide

## What You've Got

This package contains a **professional WhatsApp bulk messaging platform** built with:
- ✅ PHP 8.1+ using MVP architecture
- ✅ Complete database schema (11 tables)
- ✅ Installation wizard
- ✅ CFA pricing (Cameroon market)
- ✅ MTN & Orange Money integration ready
- ✅ WhatsApp Business API integration ready
- ✅ Core framework (Router, Database, MVC)
- ✅ Example implementations (User model, Auth controller)

## What's Included

```
sender-app/
├── Core Framework ✅ (100% Complete)
│   ├── Database, Router, MVC base classes
│   ├── Session management
│   └── View rendering
├── Configuration ✅ (100% Complete)
│   ├── App, Database, WhatsApp, Payment configs
│   └── Feature toggles
├── Database Schema ✅ (100% Complete)
│   └── Full schema with 11 tables
├── Installation System ✅ (100% Complete)
│   └── Visual wizard with all steps
├── Examples ✅ (Provided)
│   ├── User Model with authentication
│   └── Auth Controller (login, register, password reset)
└── Documentation ✅ (Complete)
    ├── README.md
    ├── PROJECT_STRUCTURE.md
    ├── IMPLEMENTATION_GUIDE.md
    └── DEVELOPMENT_ROADMAP.md
```

## 30-Second Setup

### 1. Extract Package
```bash
tar -xzf sender-app-complete.tar.gz
cd sender-app
```

### 2. Install Dependencies
```bash
composer install
```

### 3. Set Permissions
```bash
chmod -R 755 storage public/uploads
```

### 4. Run Installation
Visit: `http://yourdomain.com/install/`

The wizard will:
- Check system requirements
- Set up database
- Configure WhatsApp API
- Create admin account

### 5. Configure Cron Job
```bash
crontab -e
# Add this line:
*/1 * * * * php /full/path/to/sender-app/cron/process-queue.php
```

## What You Need to Build

The framework is ready - you need to implement the business logic:

### Priority 1: Models (3-5 days)
- Contact.php
- Template.php
- Campaign.php
- Message.php
- Payment.php

### Priority 2: Controllers (5-7 days)
- DashboardController
- ContactController
- TemplateController
- CampaignController
- InboxController

### Priority 3: Views (7-10 days)
- All HTML templates
- Frontend CSS/JS

### Priority 4: Services (3-4 days)
- WhatsAppService
- QueueService
- Payment gateways

**Total Development Time: 4-6 weeks**

See `DEVELOPMENT_ROADMAP.md` for detailed breakdown.

## How the Framework Works

### 1. Routing (public/index.php)
```php
$router->get('/contacts', [ContactController::class, 'index']);
```

### 2. Controllers (app/controllers/)
```php
class ContactController extends Controller {
    public function index() {
        $contacts = (new Contact())->all();
        return $this->render('contacts.index', ['contacts' => $contacts]);
    }
}
```

### 3. Models (app/models/)
```php
class Contact extends Model {
    protected $table = 'contacts';
    protected $fillable = ['phone', 'name', 'email'];
}
```

### 4. Views (app/views/)
```php
<h1><?php echo $this->escape($title); ?></h1>
```

## WhatsApp API Setup

### Get Your Credentials

1. **Meta Business Account**: https://business.facebook.com/
2. **Create App**: https://developers.facebook.com/
3. **Add WhatsApp Product**
4. **Get Credentials**:
   - Phone Number ID
   - Business Account ID
   - Access Token
   - Webhook Verify Token (auto-generated during install)

### Configure Webhook
- URL: `https://yourdomain.com/public/webhook/whatsapp`
- Token: From installation
- Events: messages, message_status

## Payment Integration

### Plans (in CFA)
- **Starter**: 39,000 FCFA/month
- **Professional**: 99,000 FCFA/month
- **Business**: 199,000 FCFA/month
- **Enterprise**: Custom

### Gateways
1. **MTN Mobile Money** - Configure in config/payment.php
2. **Orange Money** - Configure in config/payment.php
3. **Bank Transfer** - Manual verification by admin

## File Structure You'll Create

```
app/
├── controllers/          ⏳ YOU BUILD THIS
│   ├── DashboardController.php
│   ├── ContactController.php
│   ├── TemplateController.php
│   ├── CampaignController.php
│   ├── InboxController.php
│   ├── AnalyticsController.php
│   ├── SettingsController.php
│   ├── BillingController.php
│   ├── AdminController.php
│   └── WebhookController.php
├── models/              ⏳ YOU BUILD THIS
│   ├── Contact.php
│   ├── Group.php
│   ├── Template.php
│   ├── Campaign.php
│   ├── Message.php
│   ├── Conversation.php
│   ├── Payment.php
│   └── Subscription.php
├── views/               ⏳ YOU BUILD THIS
│   ├── layouts/
│   ├── auth/
│   ├── dashboard/
│   ├── contacts/
│   ├── templates/
│   ├── campaigns/
│   ├── inbox/
│   ├── analytics/
│   └── settings/
└── services/            ⏳ YOU BUILD THIS
    ├── WhatsAppService.php
    ├── QueueService.php
    ├── MTNMomoService.php
    ├── OrangeMoneyService.php
    └── ImportService.php
```

## Testing Your Installation

### 1. Check Installation
Visit: `http://yourdomain.com/install/`

### 2. Test Database
```php
php -r "require 'core/Database.php'; 
        use Core\Database; 
        $db = Database::getInstance(); 
        echo 'Connected!';"
```

### 3. Test WhatsApp API
Create a test script:
```php
$ch = curl_init("https://graph.facebook.com/v21.0/{phone-number-id}/messages");
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer {access-token}",
    "Content-Type: application/json"
]);
// ... send test message
```

## Common Issues

### "Class not found"
```bash
composer dump-autoload
```

### Database connection failed
Check config/database.php credentials

### WhatsApp API errors
- Verify credentials in config/whatsapp.php
- Check phone number is registered
- Ensure SSL is enabled

### Queue not processing
- Verify cron job is running: `crontab -l`
- Check logs: `tail -f storage/logs/cron.log`

## Next Steps

1. ✅ Read `README.md` for overview
2. ✅ Read `PROJECT_STRUCTURE.md` for architecture
3. ✅ Read `IMPLEMENTATION_GUIDE.md` for features
4. 🚀 Read `DEVELOPMENT_ROADMAP.md` for build plan
5. 💻 Start coding following the roadmap

## Support

All documentation is included:
- **README.md** - Overview & installation
- **PROJECT_STRUCTURE.md** - Architecture details
- **IMPLEMENTATION_GUIDE.md** - Feature implementation
- **DEVELOPMENT_ROADMAP.md** - Step-by-step build guide

## Security Reminders

- ✅ Use HTTPS (required for WhatsApp webhooks)
- ✅ Keep .env secure (never commit to git)
- ✅ Enable CSRF protection in production
- ✅ Validate all user input
- ✅ Hash all passwords (already implemented in User model)
- ✅ Use prepared statements (implemented in Database class)

## Production Checklist

Before going live:
- [ ] Set APP_ENV=production in .env
- [ ] Set APP_DEBUG=false
- [ ] Configure error logging
- [ ] Set up SSL certificate
- [ ] Configure cron jobs
- [ ] Set file permissions correctly
- [ ] Test all payment flows
- [ ] Verify WhatsApp webhook
- [ ] Set up database backups
- [ ] Configure monitoring

## Resources

- WhatsApp Docs: https://developers.facebook.com/docs/whatsapp/cloud-api
- Meta for Developers: https://developers.facebook.com/
- PHP Documentation: https://www.php.net/
- MySQL Documentation: https://dev.mysql.com/doc/

---

**You have everything you need to build a production-ready WhatsApp bulk messaging platform!**

**Estimated Time**: 4-6 weeks of development
**Difficulty**: Intermediate PHP
**Support**: All documentation included

Good luck! 🚀
