<?php

namespace Core;

abstract class Controller
{
    protected $view;
    protected $db;
    
    public function __construct()
    {
        $this->view = new View();
        $this->db = Database::getInstance();
    }
    
    protected function render($viewPath, $data = [])
    {
        return $this->view->render($viewPath, $data);
    }
    
    protected function json($data, $statusCode = 200)
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }
    
    protected function redirect($url, $statusCode = 302)
    {
        header("Location: {$url}", true, $statusCode);
        exit;
    }
    
    protected function requireAuth()
    {
        if (!Session::has('user_id')) {
            $this->redirect('/login');
        }
    }
    
    protected function requireAdmin()
    {
        $this->requireAuth();
        
        if (Session::get('user_role') !== 'admin') {
            http_response_code(403);
            die('Access denied');
        }
    }
    
    protected function getInput($key = null, $default = null)
    {
        $input = json_decode(file_get_contents('php://input'), true) ?: $_POST;
        
        if ($key === null) {
            return $input;
        }
        
        return $input[$key] ?? $default;
    }
    
    protected function validate($data, $rules)
    {
        $errors = [];
        
        foreach ($rules as $field => $ruleSet) {
            $rulesArray = explode('|', $ruleSet);
            
            foreach ($rulesArray as $rule) {
                $ruleParts = explode(':', $rule);
                $ruleName = $ruleParts[0];
                $ruleParam = $ruleParts[1] ?? null;
                
                $value = $data[$field] ?? null;
                
                switch ($ruleName) {
                    case 'required':
                        if (empty($value)) {
                            $errors[$field][] = ucfirst($field) . ' is required';
                        }
                        break;
                        
                    case 'email':
                        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                            $errors[$field][] = ucfirst($field) . ' must be a valid email';
                        }
                        break;
                        
                    case 'min':
                        if (!empty($value) && strlen($value) < $ruleParam) {
                            $errors[$field][] = ucfirst($field) . " must be at least {$ruleParam} characters";
                        }
                        break;
                        
                    case 'max':
                        if (!empty($value) && strlen($value) > $ruleParam) {
                            $errors[$field][] = ucfirst($field) . " must not exceed {$ruleParam} characters";
                        }
                        break;
                        
                    case 'unique':
                        if (!empty($value)) {
                            $table = $ruleParam;
                            $exists = $this->db->fetch("SELECT id FROM {$table} WHERE {$field} = :value LIMIT 1", ['value' => $value]);
                            if ($exists) {
                                $errors[$field][] = ucfirst($field) . ' already exists';
                            }
                        }
                        break;
                }
            }
        }
        
        return $errors;
    }
    
    protected function flashMessage($type, $message)
    {
        Session::flash($type, $message);
    }
}
